// CPPDataWindowPlugIn.cpp
//
// This file implements an IDataWindowPlugIn-derived class.
// The plug-in implements the basic parts of a data window
// in the TLA. It has a data display window and a System
// Window Control.
// 
// The window in this sample does not show any real data.
// The purpose is to demonstrate the basics of how to 
// implement the standard parts of a data window. Any custom
// timing diagrams or data interpretations are left as
// an exercise for the plug-in developer.
/////////////////////////////////////////////////////////////

#include "StdAfx.h"
// CPPDataWindowPlugIn.cpp
//
// This file implements an IDataWindowPlugIn-derived class.
// The plug-in implements the basic parts of a data window
// in the TLA. It has a data display window and a System
// Window Control.
// 
// The window in this sample does not show any real data.
// The purpose is to demonstrate the basics of how to 
// implement the standard parts of a data window. Any custom
// timing diagrams or data interpretations are left as
// an exercise for the plug-in developer.
/////////////////////////////////////////////////////////////

#using <system.windows.forms.dll>
#using <system.drawing.dll>
#using <system.dll>
#using "TlaNetInterfaces.dll"

#include "datadisplay.h"
#include "systemwindowusercontrol.h"
#include "CPPDataWindowPlugIn.h"

using namespace System;
using namespace System::Collections;
using namespace System::Windows::Forms;
using namespace Tektronix::LogicAnalyzer::TpiNet;
using namespace Tektronix::LogicAnalyzer::Common;
using namespace CPPDataWindowPlugInSample;


// The default constructor required of all plug-ins.
CPPDataWindowPlugIn::CPPDataWindowPlugIn()
{
    m_isValid = true;
    m_isGarbage = false;
}


// Initialize() is called by the TLA immediately after the 
// plug-in is instantiated with its constructor.
bool CPPDataWindowPlugIn::Initialize(ITlaPlugInSupport* support, bool userInit)
{
	// Create the DataDisplay Form window and move it 
	// into the main window of the application .
	m_display = new DataDisplay;
	support->TopLevelFormToChild(m_display);

	// Create the control that will appear in the System Window.
	m_control = new SystemWindowUserControl;

	// Subscribe to get notified when the data window is closed.
	m_display->add_Closed(new EventHandler(this, &CPPDataWindowPlugIn::OnDisplayClosed));
	
	// Subscribe to get notified when the "Show Window" button is clicked.
	m_control->m_activateButton->add_Click(new EventHandler(this, &CPPDataWindowPlugIn::OnActivateDisplay));
	
	// Subscribe to get notified when the text box loses focus
	m_control->m_userNameBox->add_LostFocus(new EventHandler(this, &CPPDataWindowPlugIn::OnLostFocus));

	// Subscribe to get notified when keys are pressed in the text box control.
	m_control->m_userNameBox->add_KeyDown(new KeyEventHandler(this, &CPPDataWindowPlugIn::OnKeyDown));
	
	// Setup to display an intial user name. If the name is not 
	// unique among data windows in the system, the TLA will change it.
	m_userName = "Sample 1";
	UserName = m_userName;
    m_control->m_userNameBox->Text = m_userName;
    m_display->Text = m_userName;
	return true;
}


// Returns the data windows currently used by the data window.
// Since this window does not actively use system data sources,
// it returns an empty array.
IDataSource* CPPDataWindowPlugIn::GetDataSources() []
{
    return new IDataSource*[0];
}


// Method that allows the user name to be programmatically set.
void CPPDataWindowPlugIn::set_UserName(String* value)
{
	// First check that the name really changed
	if (0 != String::Compare(value, m_userName)) {
        m_userName = value;
        
		// Update the control and the data window title.
		if (m_control) {
            m_control->m_userNameBox->Text = m_userName;
        }
        if (m_display) {
            m_display->Text = m_userName;
        }

		// Send notification that the name changed.
        raise_UserNameChanged(this, EventArgs::Empty);
    }
}


// This method invalidates the plug-in and disposes of its window.
void CPPDataWindowPlugIn::Dispose()
{
    if (!m_isGarbage) {
	    if (m_display) {
		    m_display->remove_Closed(new EventHandler(this, &CPPDataWindowPlugIn::OnDisplayClosed));
		    m_display->Dispose(true);
		    m_display = 0;
	    }

	    if (m_control) 
	    {
		    m_control->m_activateButton->remove_Click(
			    new EventHandler(this, &CPPDataWindowPlugIn::OnActivateDisplay));
		    m_control->m_userNameBox->remove_KeyDown(
			    new KeyEventHandler(this, &CPPDataWindowPlugIn::OnKeyDown));
		    m_control->m_userNameBox->remove_LostFocus(
			    new EventHandler(this, &CPPDataWindowPlugIn::OnLostFocus));
		    m_control->Dispose(true);
		    m_control = 0;
	    }
        
        m_isValid = false;
        m_isGarbage = true;
        
	    raise_ValidityChanged(this, EventArgs::Empty);
    }
}


// The plug-in subscribes to DataDisplay::Closed. This event handler 
// is called when the event occurs. The plug-in responds by disposing 
// itself, which deletes the plug-in instance from the system.
void CPPDataWindowPlugIn::OnDisplayClosed(System::Object*, EventArgs*)
{
    Dispose();
}


// Changes the UserName if the Enter key is pressed in the 
// SystemWindowUserControl text box.
void CPPDataWindowPlugIn::OnKeyDown(System::Object*, KeyEventArgs* args)
{
    if (Keys::Enter == args->KeyCode) {
        UserName = m_control->m_userNameBox->Text;
    }
}


// Changes the UserName when the SystemWindowUserControl text box
// loses the input focus.
void CPPDataWindowPlugIn::OnLostFocus(System::Object*, EventArgs*)
{
    UserName = m_control->m_userNameBox->Text;
}


// Activates the data window when the user presses the "Show Window"
// button on the SystemWindowUserControl.
void CPPDataWindowPlugIn::OnActivateDisplay(System::Object*, EventArgs*)
{
    if (m_display) {
        m_display->Activate();
    }   
}

